import { API_END_POINTS } from "@/services/ApiEndpoints";
import { createAsyncThunk, createSlice } from "@reduxjs/toolkit";

const initialState = {
  video: null,
  loading: false,
  error: null,
};


export const watchTimeVideo = createAsyncThunk(
  "watchTimeVideo/watchTimeVideo",
  async (videoData, { getState, rejectWithValue }) => {
    const token = getState().auth?.token;

    if (!token) {
      return rejectWithValue("No token found");
    }

    try {
      const response = await fetch(API_END_POINTS.saveWatchTime, {
        headers: {
          Authorization: `Bearer ${token}`,
          "Content-Type": "application/json",
        },
        method: "POST",
        body: JSON.stringify(videoData),
      });

      if (!response.ok) {
        const errorMessage = await response.text();
        throw new Error(errorMessage || "Failed to fetch course progress");
      }

      return await response.json();
    } catch (error) {
      return rejectWithValue(error.message || "Unknown error occurred");
    }
  }
);

// Slice for managing video watch time state
export const videoSlice = createSlice({
  name: "watchTimeVideo",
  initialState,
  reducers: {
    watchTimeSend: (state, action) => {
      state.video = action.payload;
    },
  },
  extraReducers: (builder) => {
    builder
      // Handling the loading, success, and failure cases of the async thunk
      .addCase(watchTimeVideo.pending, (state) => {
        state.loading = true;
        state.error = null; // Reset error on new request
      })
      .addCase(watchTimeVideo.fulfilled, (state, action) => {
        state.loading = false;
        state.video = action.payload; // Set the video data on success
      })
      .addCase(watchTimeVideo.rejected, (state, action) => {
        state.loading = false;
        state.error = action.payload; // Capture error message on failure
      });
  },
});

export const { watchTimeSend } = videoSlice.actions;

export default videoSlice.reducer;